<?php

declare(strict_types=1);

namespace Dedi\SyliusContactPlugin\Controller\Admin;

use Dedi\SyliusContactPlugin\Entity\ContactFormMessageInterface;
use Dedi\SyliusContactPlugin\Factory\AnswerFactoryInterface;
use Dedi\SyliusContactPlugin\Form\AnswerType;
use Dedi\SyliusContactPlugin\Mailer\AnswerMailerInterface;
use Dedi\SyliusContactPlugin\Model\AnswerModel;
use Dedi\SyliusContactPlugin\Repository\ContactFormMessageRepository;
use Doctrine\ORM\EntityManagerInterface;
use Sylius\Component\Core\Model\AdminUserInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Workflow\WorkflowInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class ShowMessageAction extends AbstractController
{
    public function __construct(
        private readonly ContactFormMessageRepository $repository,
        private readonly WorkflowInterface $workflow,
        private readonly EntityManagerInterface $entityManager,
        private readonly AnswerFactoryInterface $answerFactory,
        private readonly TranslatorInterface $translator,
        private readonly AnswerMailerInterface $answerMailer,
    ) {
    }

    public function __invoke(Request $request): Response
    {
        $template = $this->getSyliusAttribute(
            $request,
            'template',
            '@DediSyliusContactPlugin/admin/message/show.html.twig',
        );

        /** @var ContactFormMessageInterface $message */
        $message = $this->repository->find($request->attributes->getInt('id'));
        /** @var AdminUserInterface $user */
        $user = $this->getUser();
        $answerModel = new AnswerModel($message, $user);

        $form = $this->createForm(AnswerType::class, $answerModel);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            if ($answerModel->process) {
                $this->workflow->apply($message, 'process');
            }

            $answer = $this->answerFactory->createNewFromAnswerModel($answerModel);
            $message->addAnswer($answer);

            $this->entityManager->persist($answer);
            $this->entityManager->flush();

            $this->answerMailer->send($answer);

            $this->addFlash('success', $this->translator->trans(
                $this->getSyliusAttribute(
                    $request,
                    'success_flash',
                    'dedi_sylius_contact_form.answer_success',
                ),
            ));

            // create a new form in order for the form not to be pre-filled
            $form = $this->createForm(AnswerType::class, new AnswerModel($message, $user));
        }

        return $this->render($template, [
            'resource' => $message,
            'answer' => $answerModel,
            'form' => $form->createView(),
        ]);
    }

    private function getSyliusAttribute(Request $request, string $attributeName, ?string $default): ?string
    {
        $attributes = $request->attributes->get('_sylius');

        return $attributes[$attributeName] ?? $default;
    }
}
