<?php

declare(strict_types=1);

namespace Dedi\SyliusColissimoPlugin\Webservice\Controller;

use Dedi\SyliusColissimoPlugin\Webservice\Form\Type\SearchPickupPointsType;
use Dedi\SyliusColissimoPlugin\Webservice\Request\SearchPickupPointsRequestInterface;
use Dedi\SyliusColissimoPlugin\Webservice\ValueObject\SearchParameters;
use Exception;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Order\Context\CartContextInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Exception\JsonException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class SearchPickupPointsController extends AbstractController
{
    public function __construct(
        private readonly SearchPickupPointsRequestInterface $searchPickupPointsRequest,
        private readonly SerializerInterface $serializer,
        private readonly TranslatorInterface $translator,
        private readonly LoggerInterface $logger,
        private readonly CartContextInterface $cartContext,
    ) {
    }

    public function show(string $pickupPointType): Response
    {
        $form = $this->getForm($pickupPointType);

        return $this->render('@DediSyliusColissimoPlugin/Search/_form.html.twig', [
            'form' => $form->createView(),
        ]);
    }

    public function search(Request $request): JsonResponse
    {
        $parameters = $request->request->all();
        $form = $this->getForm($parameters['dedi_colissimo_search_pickup_points']['pickupPointType']);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            /** @var SearchParameters $data */
            $data = $form->getData();

            try {
                $results = $this->searchPickupPointsRequest->search($data);
                $pickupPoints = \json_decode($this->serializer->serialize($results, 'json'), true);

                return new JsonResponse([
                    'results' => $pickupPoints,
                    'html' => $this->renderView('@DediSyliusColissimoPlugin/Search/_pickupPoints.html.twig', [
                        'pickupPoints' => $pickupPoints,
                        'pickupPointType' => $data->getPickupPointType(),
                    ]),
                ]);
            } catch (Exception $e) {
                $this->logger->error($e->getCode() . ' : ' . $e->getMessage());

                throw $e;
            }
        }

        throw new JsonException(
            $this->translator->trans('dedi_colissimo_relay.ui.search_pickup_points.error'),
        );
    }

    private function getForm(string $pickupPointType): FormInterface
    {
        /** @var OrderInterface $cart */
        $cart = $this->cartContext->getCart();
        $address = $cart->getShippingAddress();

        return $this->createForm(SearchPickupPointsType::class, new SearchParameters(), [
            'method' => Request::METHOD_POST,
            'action' => $this->generateUrl('dedi_colissimo_search_pickup_point'),
            'city' => $address->getCity(),
            'postalCode' => $address->getPostcode(),
            'pickupPointType' => $pickupPointType,
        ]);
    }
}
