<?php

declare(strict_types=1);

namespace Dedi\SyliusCmsPlugin\Twig;

use Dedi\SyliusCmsPlugin\Utils\BlockList\BlockListInterface;
use MonsieurBiz\SyliusRichEditorPlugin\Exception\UiElementNotFoundException;
use MonsieurBiz\SyliusRichEditorPlugin\Twig\RichEditorExtension;
use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;
use Twig\TwigFunction;

class UiElementRenderExtension extends AbstractExtension
{
    private readonly RichEditorExtension $richEditorExtension;

    private readonly BlockListInterface $blockList;

    public function __construct(RichEditorExtension $richEditorExtension, BlockListInterface $blockList)
    {
        $this->richEditorExtension = $richEditorExtension;
        $this->blockList = $blockList;
    }

    public function renderField(?string $content, string $rowClassName = 'row', array $context = []): string
    {
        $elements = json_decode($content, true);
        if (!\is_array($elements)) {
            return $content;
        }

        return $this->renderElements($elements, $rowClassName, $context);
    }

    public function renderElements(array $elements, string $className = 'row', array $context = []): string
    {
        $html = '';
        $totalWidth = 0;
        $containers = [[]];
        $index = 0;
        foreach ($elements as $currentIndex => $element) {
            $containers[$index][] = $element;

            $totalWidth += isset($element['data']['width']) ? (int) $element['data']['width'] : 100;
            if ($totalWidth >= 100 || $totalWidth + $this->getNextWidth($elements, $currentIndex) > 100) {
                $totalWidth = 0;
                $containers[] = [];
                ++$index;
            }
        }

        foreach ($containers as $container) {
            if (count($container) === 0) {
                continue;
            }

            $html .= sprintf('<div class="%s">', $className);
            foreach ($container as $containerElement) {
                try {
                    $html .= $this->richEditorExtension->renderElement($context, $containerElement);
                } catch (UiElementNotFoundException $e) {
                    continue;
                }
            }
            $html .= '</div>';
        }

        return $html;
    }

    private function getNextWidth(array $elements, int $currentIndex): int
    {
        if (!isset($elements[$currentIndex + 1])) {
            return 0;
        }

        if (!isset($elements[$currentIndex + 1]['data']['width'])) {
            return 100;
        }

        return (int) $elements[$currentIndex + 1]['data']['width'];
    }

    public function getTemplateElement(string $name, string $env = 'shop'): ?string
    {
        $element = $this->blockList->getElementByName($name);

        if (null !== $element && $env === 'shop') {
            return $element['shop_template'];
        }
        if (null !== $element && $env === 'admin') {
            return $element['admin_template'];
        }

        return null;
    }

    public function getFilters(): array
    {
        return [
            new TwigFilter('dedi_cms_render_field', [$this, 'renderField'], ['is_safe' => ['html']]),
        ];
    }

    public function getFunctions(): array
    {
        return [
            new TwigFunction('dedi_cms_template_element', [$this, 'getTemplateElement']),
        ];
    }
}
