<?php

declare(strict_types=1);

namespace Dedi\SyliusCmsPlugin\Sitemap;

use BitBag\SyliusCmsPlugin\Entity\PageInterface;
use BitBag\SyliusCmsPlugin\Entity\PageTranslationInterface;
use BitBag\SyliusCmsPlugin\Repository\PageRepositoryInterface;
use Doctrine\Common\Collections\Collection;
use SitemapPlugin\Factory\AlternativeUrlFactoryInterface;
use SitemapPlugin\Factory\UrlFactoryInterface;
use SitemapPlugin\Model\ChangeFrequency;
use SitemapPlugin\Model\UrlInterface;
use SitemapPlugin\Provider\UrlProviderInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Locale\Context\LocaleContextInterface;
use Sylius\Component\Locale\Model\LocaleInterface;
use Sylius\Component\Resource\Model\TranslationInterface;
use Symfony\Component\Routing\RouterInterface;

final class PageUrlProvider implements UrlProviderInterface
{
    private readonly PageRepositoryInterface $pageRepository;

    private readonly RouterInterface $router;

    private readonly UrlFactoryInterface $sitemapUrlFactory;

    private readonly LocaleContextInterface $localeContext;

    private readonly AlternativeUrlFactoryInterface $urlAlternativeFactory;

    private ChannelInterface $channel;

    /** @var array<string|null> */
    private array $channelLocaleCodes;

    public function __construct(
        PageRepositoryInterface $pageRepository,
        RouterInterface $router,
        UrlFactoryInterface $sitemapUrlFactory,
        LocaleContextInterface $localeContext,
        AlternativeUrlFactoryInterface $urlAlternativeFactory,
    ) {
        $this->pageRepository = $pageRepository;
        $this->router = $router;
        $this->sitemapUrlFactory = $sitemapUrlFactory;
        $this->localeContext = $localeContext;
        $this->urlAlternativeFactory = $urlAlternativeFactory;
    }

    public function getName(): string
    {
        return 'cms_pages';
    }

    public function generate(ChannelInterface $channel): iterable
    {
        $this->channel = $channel;
        $this->channelLocaleCodes = [];

        $urls = [];

        foreach ($this->getPages() as $page) {
            $urls[] = $this->createPageUrl($page);
        }

        return $urls;
    }

    private function getTranslations(PageInterface $page): Collection
    {
        return $page->getTranslations()->filter(fn (TranslationInterface $translation) => $this->localeInLocaleCodes($translation));
    }

    private function localeInLocaleCodes(TranslationInterface $translation): bool
    {
        return in_array($translation->getLocale(), $this->getLocaleCodes(), true);
    }

    private function getPages(): iterable
    {
        return $this->pageRepository->findEnabled(true);
    }

    private function getLocaleCodes(): array
    {
        if ($this->channelLocaleCodes === []) {
            $this->channelLocaleCodes = $this->channel->getLocales()->map(fn (LocaleInterface $locale): ?string => $locale->getCode())->toArray();
        }

        return $this->channelLocaleCodes;
    }

    private function createPageUrl(PageInterface $page): UrlInterface
    {
        $pageUrl = $this->sitemapUrlFactory->createNew('');

        $pageUrl->setChangeFrequency(ChangeFrequency::daily());
        $pageUrl->setPriority(0.7);

        if (null !== $page->getUpdatedAt()) {
            $pageUrl->setLastModification($page->getUpdatedAt());
        } elseif (null !== $page->getCreatedAt()) {
            $pageUrl->setLastModification($page->getCreatedAt());
        }

        /** @var PageTranslationInterface $translation */
        foreach ($this->getTranslations($page) as $translation) {
            $locale = $translation->getLocale();

            if ($locale === null) {
                continue;
            }

            if (!$this->localeInLocaleCodes($translation)) {
                continue;
            }

            $location = $this->router->generate('bitbag_sylius_cms_plugin_shop_page_show', [
                'slug' => $translation->getSlug(),
                '_locale' => $locale,
            ]);

            if ($locale === $this->localeContext->getLocaleCode()) {
                $pageUrl->setLocation($location);

                continue;
            }

            $pageUrl->addAlternative($this->urlAlternativeFactory->createNew($location, $locale));
        }

        return $pageUrl;
    }
}
