<h1>Features
</h1>

### Attaching a block to an entity

You can quickly attach a block to an entity. This allows to use the CMS editing capabilities of the plugin to manage the content associated to a given entity. 

For example, you can attach a CMS Block to a Taxon and display the block on the Taxon Page. 

#### Make the entity BlockAware

Implement your interface

    ```php
    <?php
    
    declare(strict_types=1);

    namespace App\Entity;

    use Dedi\SyliusCmsPlugin\Entity\BlockAwareInterface;
    use Sylius\Component\Core\Model\TaxonInterface as BaseTaxonInterface;
    
    interface TaxonInterface extends BaseTaxonInterface, BlockAwareInterface
    ```

Implement your entity
    
    ```php
    <?php
    
    declare(strict_types=1);

    namespace App\Entity;
    
    use Dedi\SyliusCmsPlugin\Entity\BlockAwareTrait;
    use Sylius\Component\Core\Model\Taxon as BaseTaxon;
    
    class Taxon extends BaseTaxon implements TaxonInterface
    {
        use BlockAwareTrait;
    
        public function getBlockCodePrefix(): string
        {
            return 'TAXON';
        }
    }
    ```
    
Don't forget to tell Sylius to use your new entities

```yaml
# config/packages/_sylius.yaml
sylius_taxonomy:
    resources:
        taxon:
            classes:
                model: App\Entity\Taxon
                interface: App\Entity\TaxonInterface
```

    
Note : the `getBlockCodePrefix` function is used to generate the code for the associated Block (see `BlockAwareTrait::generateBlockCode`). The code follows the pattern `[codePrefix]_[entityCode]`. This means that your entity should be `CodeAware` as defined in the `BlockAwareInterface`

### Update your database 

```bash
$ bin/console doctrine:migrations:diff
$ bin/console doctrine:migrations:migrate
```

### Add the block management section to the entity form

Define a type extension for your entity's form and use the `BlockAwareTraitType` to add the block admin. 

    ```php
    <?php
    
    declare(strict_types=1);
    
    namespace App\Form\Extension;
    
    use Dedi\SyliusCmsPlugin\Form\Type\BlockAwareType;
    use Sylius\Bundle\TaxonomyBundle\Form\Type\TaxonType;
    use Symfony\Component\Form\AbstractTypeExtension;
    use Symfony\Component\Form\FormBuilderInterface;
    
    class TaxonTypeExtension extends AbstractTypeExtension
    {
        use BlockAwareTraitType;

        public function buildForm(FormBuilderInterface $builder, array $options): void
        {
            ... 
            
            $this->buildBlockAwareForm($builder, $options);    
            
            ...
        }
    
        public static function getExtendedTypes(): iterable
        {
            return [TaxonType::class];
        }
    }
    ```
This trait adds a block field and an event listener to update the block code based on the associated entity's code. 
    
### Render the block management section

Use the form_theme provided by the plugin and display the new field as a row

```twig
{# templates/bundles/SyliusAdminBundle/Taxon/_form.html.twig #}

...

{% form_theme form '@DediSyliusCmsPlugin/Admin/BlockAware/Form/theme.html.twig' %}

{{ form_row(form.block) }}

...

```

### Render the content

```twig
    {{ bitbag_cms_render_block(taxon.block.code) }}
```

### Customize the admin form

To customize the admin form, you need to declare the `admin_form` key in the UI Element configuration

```yaml
        dedi_sylius_cms_plugin.product_collection_block:
          ...
          templates:
            admin_form: '@DediSyliusCmsPlugin/Form/UiElement/ProductCollection/form.html.twig'
          ...
```

In the form template, you need to extend the original form template `@MonsieurBizSyliusRichEditorPlugin/Admin/form.html.twig`

You can then override the `form_content` block and render your form

```twig
{% extends '@MonsieurBizSyliusRichEditorPlugin/Admin/form.html.twig' %}

{% block form_content %}
    {{ form(form, {'action': url('monsieurbiz_richeditor_admin_form_submit', {'code': uiElement.code, 'isEdition': isEdition})}) }}
{% endblock %}
```

#### Collection customization

You need to create a form theme specific for the form you are customizing and tell your form to use this theme. 

```twig
# src/Resources/views/Form/UiElement/ProductCollection/theme.html.twig
{% extends '@DediSyliusCmsPlugin/Form/theme.html.twig' %}
```

```twig
# src/Resources/views/Form/UiElement/ProductCollection/form.html.twig
{% extends '@MonsieurBizSyliusRichEditorPlugin/Admin/form.html.twig' %}

{% block form_theme %}
    {% form_theme form '@DediSyliusCmsPlugin/Form/UiElement/ProductCollection/theme.html.twig' %}
{% endblock %}
```

You can now use the theme to customize parts of the template (see [https://symfony.com/doc/current/form/form_themes.html](https://symfony.com/doc/current/form/form_themes.html))

To customize collections, you need to override the collection widget and the macro used to render the collection entries (this is because `SyliusUiBundle` does not call `form_row` to render the collection entries )

The block `collection_widget` is called whenever a CollectionType form is rendered. It uses the `collection_item` macro to render the collection entries. See `src/Resources/views/Form/theme.html.twig`

We use the [Form Fragment Naming convention](https://symfony.com/doc/current/form/form_themes.html#form-fragment-naming) to override the collection rendering for our form collection

The block responsible for the collection rendering is named with the following pattern : `_[form_type_name]_[collection_field_name]_widget`

```twig
# src/Resources/views/Form/UiElement/ProductCollection/theme.html.twig
{% extends '@MonsieurBizSyliusRichEditorPlugin/Form/theme.html.twig' %}

{% block _product_collection_collection_widget %}
    # Overrides the block collection_widget that renders the ProductCollectionType.collection field

    {% import _self as self %}
    ...     
        {% for child in form %}
            {{ self.product_collection_row(child, allow_delete, button_delete_label, loop.index0) }}
        {% endfor %}
    ...
{% endblock %}

{% macro product_collection_row(form, allow_delete, button_delete_label, name) %}
    ...
        {{ form_widget(form) }}
    ...
{% endmacro %}
```

You should always copy the `collection_widget` block and the `collection_item` macro from `src/Resources/views/Form/theme.html.twig` as a starting point for your custom block and macro

#### Automated product block

The automated product block displays products based on rules chosen by the user when creating the block. You can easily add new rule by following those steps

Create the rule implementing `ProductAutoRuleInterface`. By implementing `ProductAutoRuleInterface`, the rule will automatically be added to the rule choice list in the block creation form. 

```php
<?php

declare(strict_types=1);

namespace App\SyliusCmsPlugin\Utils\ProductAuto\Rules;

use Dedi\SyliusCmsPlugin\Utils\ProductAuto\ProductAutoRuleInterface;

class MyRule implements ProductAutoRuleInterface
{
    public function getProducts(int $numberOfProducts, array $configuration): array
    {
        // your logic here        
    }

    public function getType(): string
    {
        return 'my_rule';
    }

    public function getConfigurationFormClass(): ?string
    {
        return App\SyliusCmsPlugin\Form\MyRuleConfigurationFormType::class;
    }
}
```

The function `getType` is used to :
    - add an entry in the rule choice type of the automated product block creation form ()
    - query translations for the entry `dedi_cms.ui_element.product_auto.rule.[getType]`
    - define the name of the form_row that will be added if your rule has specific configuration (eg : `{{ form_row(form.[getType]_configuration) }}`

The function `getConfigurationFormClass` returns the configuration FormType that adds specific form fields to the block configuration. 

For example, the `LatestByTaxon` rule needs to allow the user to choose a Taxon. We use `LatestByTaxonRuleConfigurationType` (returned by `getConfigurationFormClass`) to add a TaxonAutocompleteFormType to the form.

The extra form values are then passed to the rule in the `configuration` array of the `getProducts` function

The function `getProducts` is where the magic happens : retrieve the products that you want to display and returned them in an array. That's it !