<?php

declare(strict_types=1);

namespace Dedi\SyliusCmsPlugin\Utils\ProductAuto\Rules;

use Dedi\SyliusCmsPlugin\Form\Type\ProductAutoRule\LatestByTaxonRuleConfigurationType;
use Dedi\SyliusCmsPlugin\Repository\ProductRepositoryInterface;
use Dedi\SyliusCmsPlugin\Utils\ProductAuto\ProductAutoRuleInterface;
use LogicException;
use Sylius\Component\Channel\Context\ChannelContextInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\TaxonInterface;
use Sylius\Component\Locale\Context\LocaleContextInterface;
use Sylius\Component\Taxonomy\Repository\TaxonRepositoryInterface;

class LatestByTaxon implements ProductAutoRuleInterface
{
    private readonly ChannelContextInterface $channelContext;

    private readonly LocaleContextInterface $localeContext;

    private readonly TaxonRepositoryInterface $taxonRepository;

    private readonly ProductRepositoryInterface $productRepository;

    private readonly bool $includeAllDescendants;

    public function __construct(
        ChannelContextInterface $channelContext,
        LocaleContextInterface $localeContext,
        TaxonRepositoryInterface $taxonRepository,
        ProductRepositoryInterface $productRepository,
        bool $includeAllDescendants,
    ) {
        $this->channelContext = $channelContext;
        $this->localeContext = $localeContext;
        $this->taxonRepository = $taxonRepository;
        $this->productRepository = $productRepository;
        $this->includeAllDescendants = $includeAllDescendants;
    }

    public function getProducts(int $numberOfProducts, array $configuration): array
    {
        if (!array_key_exists('taxon', $configuration)) {
            throw new LogicException('Configuration key taxon must be defined');
        }

        $taxon = $this->taxonRepository->findOneBy([
            'id' => $configuration['taxon'],
        ]);

        if (!$taxon instanceof TaxonInterface) {
            throw new LogicException(sprintf('Taxon with id %s not found', $configuration['taxon']));
        }

        /** @var ChannelInterface $channel */
        $channel = $this->channelContext->getChannel();

        return $this->productRepository->findLatestByTaxon(
            $channel,
            $taxon,
            $this->localeContext->getLocaleCode(),
            $numberOfProducts,
            $this->includeAllDescendants,
        );
    }

    public function getType(): string
    {
        return 'latest_by_taxon';
    }

    public function getConfigurationFormClass(): ?string
    {
        return LatestByTaxonRuleConfigurationType::class;
    }
}
