<?php

declare(strict_types=1);

namespace Dedi\SyliusCmsPlugin\Form\Type\UiElement\Base;

use Dedi\SyliusCmsPlugin\Form\Type\UiElement\AbstractUiElementType;
use MonsieurBiz\SyliusRichEditorPlugin\Form\Type\UiElement\ImageType;
use MonsieurBiz\SyliusRichEditorPlugin\Form\Type\UiElement\VideoType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\FileType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Validator\Constraints\File;
use Symfony\Component\Validator\Constraints\Image;

class MediaType extends AbstractUiElementType
{
    private const TYPE_PDF = 'pdf';

    private const TYPE_IMAGE = 'image';

    private const TYPE_YOUTUBE = 'youtube';

    private const TYPE_VIDEO = 'video';

    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        parent::buildForm($builder, $options);

        $builder
            ->add('type', ChoiceType::class, [
                'label' => 'dedi_cms.ui_element.media.field.type',
                'choices' => [
                    'dedi_cms.ui_element.media.field.pdf' => self::TYPE_PDF,
                    'dedi_cms.ui_element.media.field.image' => self::TYPE_IMAGE,
                    'dedi_cms.ui_element.media.field.youtube' => self::TYPE_YOUTUBE,
                    'dedi_cms.ui_element.media.field.video' => self::TYPE_VIDEO,
                ],
                'attr' => [
                    'data-block-select-type' => 'data-block-select-type',
                ],
            ])
            ->add('pdf', FileType::class, [
                'label' => 'dedi_cms.ui_element.media.field.pdf',
                'data_class' => null,
                'constraints' => [
                    new File([
                        'mimeTypes' => [
                            'application/pdf',
                            'application/x-pdf',
                        ],
                        'mimeTypesMessage' => 'app.form.error_pdf',
                    ]),
                ],
                'attr' => [
                    'accept' => 'application/pdf,application/x-pdf',
                    'data-block-type' => 'pdf',
                ],
                'row_attr' => [
                    'data-test' => 'test',
                ],
            ])
            ->add('image', ImageType::class, [
                'label' => false,
                'attr' => [
                    'data-block-type' => 'image',
                ],
                'embedded' => true,
            ])
            ->add('video', VideoType::class, [
                'label' => false,
                'attr' => [
                    'data-block-type' => 'video',
                ],
                'required' => false,
                'embedded' => true,
            ])
            ->add('youtubeId', TextType::class, [
                'label' => 'dedi_cms.ui_element.media.youtube_id',
                'attr' => [
                    'data-block-type' => 'youtube',
                ],
            ])
            ->add('youtubeImage', FileType::class, [
                'label' => 'dedi_cms.ui_element.media.youtube_image',
                'required' => false,
                'data_class' => null,
                'attr' => [
                    'data-image' => 'true',
                    'data-block-type' => 'youtube',
                ],
                'constraints' => [
                    new Image(),
                ],
            ])
        ;

        $builder
            ->get('image')
            ->remove('title')
        ;

        // this event listener removes unused data
        // for instance when a media of type image is submitted, it makes sure pdf and youtube data is not saved
        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event): void {
            $data = $event->getData();
            $form = $event->getForm();

            switch ($data['type']) {
                case 'pdf':
                    $form->remove('video');
                    $form->remove('image');
                    $form->remove('youtubeId');
                    $form->remove('youtubeImage');
                    unset($data['video'], $data['image'], $data['youtubeId'], $data['youtubeImage']);

                    break;
                case 'youtube':
                    $form->remove('video');
                    $form->remove('image');
                    $form->remove('pdf');
                    unset($data['video'], $data['image'], $data['pdf']);

                    break;
                case 'video':
                    $form->remove('pdf');
                    $form->remove('image');
                    $form->remove('youtubeId');
                    $form->remove('youtubeImage');
                    unset($data['pdf'], $data['image'], $data['youtubeId'], $data['youtubeImage']);

                    break;
                case 'image':
                    $form->remove('pdf');
                    $form->remove('video');
                    $form->remove('youtubeId');
                    $form->remove('youtubeImage');
                    unset($data['pdf'], $data['video'], $data['youtubeId'], $data['youtubeImage']);

                    break;
            }

            $event->setData($data);
        });

        // add another event listener, but this time on the image field
        // its role is to remove the constraint that are added by the event listener on the image field in ImageType
        // it can't be done in the event listener above as they are not listening for the same thing
        $builder->get('image')->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event): void {
            $data = $event->getData();

            // this means we should remove the image not null constraint
            if ($data['image'] === false) {
                $event->getForm()->add('image');
            }
        });
    }
}
