<?php

declare(strict_types=1);

namespace Dedi\SyliusCmsPlugin\Sitemap;

use BitBag\SyliusCmsPlugin\Entity\SectionInterface;
use BitBag\SyliusCmsPlugin\Entity\SectionTranslationInterface;
use BitBag\SyliusCmsPlugin\Repository\SectionRepositoryInterface;
use Doctrine\Common\Collections\Collection;
use SitemapPlugin\Factory\AlternativeUrlFactoryInterface;
use SitemapPlugin\Factory\UrlFactoryInterface;
use SitemapPlugin\Model\ChangeFrequency;
use SitemapPlugin\Model\UrlInterface;
use SitemapPlugin\Provider\UrlProviderInterface;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Locale\Context\LocaleContextInterface;
use Sylius\Component\Locale\Model\LocaleInterface;
use Sylius\Component\Resource\Model\TranslationInterface;
use Symfony\Component\Routing\RouterInterface;

final class SectionUrlProvider implements UrlProviderInterface
{
    private readonly SectionRepositoryInterface $sectionRepository;

    private readonly RouterInterface $router;

    private readonly UrlFactoryInterface $sitemapUrlFactory;

    private readonly LocaleContextInterface $localeContext;

    private readonly AlternativeUrlFactoryInterface $urlAlternativeFactory;

    private ChannelInterface $channel;

    /** @var array<string|null> */
    private array $channelLocaleCodes;

    public function __construct(
        SectionRepositoryInterface $sectionRepository,
        RouterInterface $router,
        UrlFactoryInterface $sitemapUrlFactory,
        LocaleContextInterface $localeContext,
        AlternativeUrlFactoryInterface $urlAlternativeFactory,
    ) {
        $this->sectionRepository = $sectionRepository;
        $this->router = $router;
        $this->sitemapUrlFactory = $sitemapUrlFactory;
        $this->localeContext = $localeContext;
        $this->urlAlternativeFactory = $urlAlternativeFactory;
    }

    public function getName(): string
    {
        return 'cms_sections';
    }

    public function generate(ChannelInterface $channel): iterable
    {
        $this->channel = $channel;
        $this->channelLocaleCodes = [];

        $urls = [];

        foreach ($this->getSections() as $section) {
            $urls[] = $this->createSectionUrl($section);
        }

        return $urls;
    }

    private function getTranslations(SectionInterface $section): Collection
    {
        return $section->getTranslations()->filter(fn (TranslationInterface $translation) => $this->localeInLocaleCodes($translation));
    }

    private function localeInLocaleCodes(TranslationInterface $translation): bool
    {
        return in_array($translation->getLocale(), $this->getLocaleCodes(), true);
    }

    private function getSections(): iterable
    {
        return $this->sectionRepository->findAll();
    }

    private function getLocaleCodes(): array
    {
        if ($this->channelLocaleCodes === []) {
            $this->channelLocaleCodes = $this->channel->getLocales()->map(fn (LocaleInterface $locale): ?string => $locale->getCode())->toArray();
        }

        return $this->channelLocaleCodes;
    }

    private function createSectionUrl(SectionInterface $section): UrlInterface
    {
        $url = $this->sitemapUrlFactory->createNew('');

        $url->setChangeFrequency(ChangeFrequency::daily());
        $url->setPriority(0.7);

        /** @var SectionTranslationInterface $translation */
        foreach ($this->getTranslations($section) as $translation) {
            $locale = $translation->getLocale();

            if ($locale === null) {
                continue;
            }

            if (!$this->localeInLocaleCodes($translation)) {
                continue;
            }

            $location = $this->router->generate('bitbag_sylius_cms_plugin_shop_section_show', [
                'code' => $section->getCode(),
                '_locale' => $translation->getLocale(),
            ]);

            if ($locale === $this->localeContext->getLocaleCode()) {
                $url->setLocation($location);

                continue;
            }

            $url->addAlternative($this->urlAlternativeFactory->createNew($location, $locale));
        }

        return $url;
    }
}
