<?php

declare(strict_types=1);

namespace Dedi\SyliusCmsPlugin\Repository;

use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\QueryBuilder;
use LogicException;
use Sylius\Component\Core\Model\ChannelInterface;
use Sylius\Component\Core\Model\OrderItemInterface;
use Sylius\Component\Core\Model\TaxonInterface;
use Sylius\Component\Core\OrderPaymentStates;

trait ProductRepositoryTrait
{
    protected function createSearchQueryBuilder(ChannelInterface $channel, string $locale, string $alias = 'product'): QueryBuilder
    {
        return $this
            ->createQueryBuilder($alias)
            ->addSelect('translation')
            ->innerJoin(sprintf('%s.variants', $alias), 'variant')
            ->innerJoin(sprintf('%s.translations', $alias), 'translation', 'WITH', 'translation.locale = :locale')
            ->andWhere(sprintf('%s.enabled = :enabled', $alias))
            ->andWhere(sprintf(':channel MEMBER OF %s.channels', $alias))
            ->setParameter('channel', $channel)
            ->setParameter('locale', $locale)
            ->setParameter('enabled', true)
        ;
    }

    public function findBestSellerByChannel(ChannelInterface $channel, string $locale, int $count): array
    {
        if (!$this instanceof EntityRepository) {
            throw new LogicException('Dedi\SyliusCmsPlugin\Repository\ProductRepositoryTrait should be used on an EntityRepository');
        }

        return $this->createSearchQueryBuilder($channel, $locale)
            ->innerJoin(OrderItemInterface::class, 'orderItem', 'WITH', 'variant = orderItem.variant')
            ->innerJoin('orderItem.order', 'ord', 'WITH', 'ord.channel = :channel')
            ->andWhere('ord.checkoutCompletedAt BETWEEN :last30days AND :today')
            ->andWhere('ord.paymentState = :paymentState')
            ->setParameter('today', date('Y-m-d', strtotime('+1 day')))
            ->setParameter('last30days', date('Y-m-d', strtotime('-30 days')))
            ->setParameter('paymentState', OrderPaymentStates::STATE_PAID)
            ->groupBy('product')
            ->orderBy('COUNT(DISTINCT orderItem.order)', 'DESC')
            ->setMaxResults($count)
            ->getQuery()
            ->getResult()
        ;
    }

    public function findRandomByChannel(ChannelInterface $channel, string $locale, int $count): array
    {
        if (!$this instanceof EntityRepository) {
            throw new LogicException('Dedi\SyliusCmsPlugin\Repository\ProductRepositoryTrait should be used on an EntityRepository');
        }

        return $this->createSearchQueryBuilder($channel, $locale)
            ->addOrderBy('RAND()')
            ->groupBy('product')
            ->setMaxResults($count)
            ->getQuery()
            ->getResult()
        ;
    }

    public function findLatestByTaxon(ChannelInterface $channel, TaxonInterface $taxon, string $locale, int $count, bool $includeAllDescendants = false): array
    {
        if (!$this instanceof EntityRepository) {
            throw new LogicException('Dedi\SyliusCmsPlugin\Repository\ProductRepositoryTrait should be used on an EntityRepository');
        }

        $queryBuilder = $this->createSearchQueryBuilder($channel, $locale)
            ->addOrderBy('product.createdAt', 'DESC')
            ->innerJoin('product.productTaxons', 'productTaxon')
        ;

        if ($includeAllDescendants) {
            $queryBuilder
                ->innerJoin('productTaxon.taxon', 'taxon')
                ->andWhere('taxon.left >= :taxonLeft')
                ->andWhere('taxon.right <= :taxonRight')
                ->andWhere('taxon.root = :taxonRoot')
                ->setParameter('taxonLeft', $taxon->getLeft())
                ->setParameter('taxonRight', $taxon->getRight())
                ->setParameter('taxonRoot', $taxon->getRoot())
            ;
        } else {
            $queryBuilder
                ->andWhere('productTaxon.taxon = :taxon')
                ->setParameter('taxon', $taxon)
            ;
        }

        return $queryBuilder
            ->groupBy('product')
            ->setMaxResults($count)
            ->getQuery()
            ->getResult()
        ;
    }
}
