<?php

declare(strict_types=1);

namespace Dedi\SyliusAvisVerifiesPlugin\Client;

use Dedi\SyliusAvisVerifiesPlugin\Client\Factory\CredentialsFactoryInterface;
use Dedi\SyliusAvisVerifiesPlugin\Exception\AuthenticationException;
use Psr\Log\LoggerInterface;
use Symfony\Contracts\HttpClient\Exception\ClientExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface as BaseHttpClientInterface;
use Symfony\Contracts\HttpClient\ResponseInterface;

final class HttpClient implements HttpClientInterface
{
    private ?string $accessToken = null;

    private \DateTimeInterface $expiresAt;

    public function __construct(
        private BaseHttpClientInterface $client,
        private BaseHttpClientInterface $authClient,
        private readonly CredentialsFactoryInterface $credentialsFactory,
        private readonly string $baseAuthUrl,
        private readonly string $baseUrl,
        private readonly LoggerInterface $logger,
    ) {
        $this->authClient = $this->authClient->withOptions([
           'base_uri' => $this->baseAuthUrl,
        ]);
        $this->client = $this->client->withOptions([
           'base_uri' => $this->baseUrl,
        ]);
    }

    private function authenticate(): ?string
    {
        if (null !== $this->accessToken && $this->expiresAt > new \DateTime()) {
            return $this->accessToken;
        }

        $credentials = $this->credentialsFactory->create();

        try {
            $response = $this->authClient->request('POST', 'realms/skeepers/protocol/openid-connect/token', [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded',
                    'authorization' => 'Basic ' . base64_encode($credentials->getClientId() . ':' . $credentials->getSecretKey()),
                ],
                'body' => [
                    'grant_type' => 'client_credentials',
                    'scope' => 'openid',
                ],
            ]);

            $result = $response->toArray(false);

            if (200 !== $response->getStatusCode()) {
                throw new AuthenticationException($result['error_description'], $response->getStatusCode());
            }

            $this->accessToken = $result['access_token'];
            $this->expiresAt = (new \DateTimeImmutable())->add(new \DateInterval('PT' . $result['expires_in'] . 'S'));
        } catch (ClientExceptionInterface|TransportExceptionInterface $e) {
            $this->logger->error($e->getMessage(), ['exception' => $e]);
        }

        return $this->accessToken;
    }

    public function post(string $url, array $body): ResponseInterface
    {
        $credentials = $this->credentialsFactory->create();

        $url = str_replace('{website_id}', $credentials->getWebsiteId(), $url);

        return $this->client->request('POST', $url, [
            'auth_bearer' => $this->authenticate(),
            'json' => $body,
        ]);
    }
}
