<?php

declare(strict_types=1);

namespace Dedi\SyliusAvisVerifiesPlugin\Console;

use Dedi\SyliusAvisVerifiesPlugin\Request\SendOrderRequestInterface;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Order\Repository\OrderRepositoryInterface;
use Sylius\Component\Shipping\Model\ShipmentInterface;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

#[AsCommand(
    name: 'dedi-sylius-avis-verifies:export-orders',
    description: 'Send shipped orders to Avis Verifies.',
)]
class ExportOrdersCommand extends Command
{
    public function __construct(
        private readonly SendOrderRequestInterface $sendOrderRequest,
        private readonly OrderRepositoryInterface $orderRepository,
        private readonly LoggerInterface $logger,
    ) {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->addArgument('orderId', InputArgument::REQUIRED, 'Id of order to export')
        ;
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        $orderId = (int) $input->getArgument('orderId');

        /** @var OrderInterface|null $order */
        $order = $this->orderRepository->findOneBy([
            'shippingState' => ShipmentInterface::STATE_SHIPPED,
            'id' => $orderId,
        ]);

        if (null === $order) {
            $message = sprintf('Shipped order with id %d not found', $orderId);
            $io->error($message);
            $this->logger->error($message);

            return Command::FAILURE;
        }

        $this->sendOrderRequest->send($order);

        $io->success(sprintf('Successfully sent order %d', (int) $order->getId()));

        return Command::SUCCESS;
    }
}
