<?php

declare(strict_types=1);

namespace Dedi\SyliusAvisVerifiesPlugin\Request;

use Dedi\SyliusAvisVerifiesPlugin\Client\HttpClientInterface;
use Dedi\SyliusAvisVerifiesPlugin\Exception\PostOrderException;
use Dedi\SyliusAvisVerifiesPlugin\Model\AvisVerifiesAwareInterface;
use Liip\ImagineBundle\Service\FilterService;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\ImageInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\OrderItemInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Symfony\Component\Routing\RouterInterface;
use Webmozart\Assert\Assert;

final class SendOrderRequest implements SendOrderRequestInterface
{
    public function __construct(
        private readonly HttpClientInterface $client,
        private readonly RouterInterface $router,
        private readonly LoggerInterface $logger,
        private readonly FilterService $imageFilterService,
        private readonly int $delay = 0,
    ) {
    }

    public function send(OrderInterface $order): void
    {
        try {
            $body = [$this->createMessage($order)];

            $response = $this->client->post('purchase-event/websites/{website_id}/purchase_events/bulk_sync_insert', $body);

            switch ($response->getStatusCode()) {
                case 201:
                case 206:
                    return;
                case 400:
                default:
                    throw new PostOrderException(\json_encode($body, \JSON_THROW_ON_ERROR));
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage(), ['exception' => $e]);
        }
    }

    private function createMessage(OrderInterface $order): array
    {
        $products = array_map(
            function (OrderItemInterface $orderItem) use ($order): array {
                /** @var ProductInterface|AvisVerifiesAwareInterface $product */
                $product = $orderItem->getProduct();

                Assert::isInstanceOf($product, ProductInterface::class);
                Assert::isInstanceOf($product, AvisVerifiesAwareInterface::class);

                $productData = [
                    'product_ref' => [
                        'reference' => $product->getId(),
                        'sku' => $orderItem->getVariant()->getCode() !== null ? $orderItem->getVariant()->getCode() : $product->getCode(),
                    ],
                    'name' => $product->getName(),
                    'product_url' => $this->router->generate(
                        'sylius_shop_product_show',
                        ['slug' => $product->getSlug(), '_locale' => $order->getLocaleCode()],
                        RouterInterface::ABSOLUTE_URL,
                    ),
                ];

                try {
                    /** @var ImageInterface|bool $image */
                    $image = $product->getImagesByType('main')->first();
                    if ($image !== false) {
                        $productData['image_url'] = $this->imageFilterService->getUrlOfFilteredImage(
                            $image->getPath(),
                            'sylius_shop_product_thumbnail',
                        );
                    }
                } catch (\Exception $e) {
                    $this->logger->error($e->getMessage(), ['exception' => $e]);
                }

                return $productData;
            },
            $order->getItems()->toArray(),
        );

        $customer = $order->getCustomer();

        return [
            'purchase_reference' => $order->getNumber(),
            'purchase_date' => $order->getCheckoutCompletedAt()?->format('Y-m-d\TH:i:s\Z'),
            'consumer' => [
                'first_name' => $customer?->getFirstName(),
                'last_name' => $customer?->getLastName(),
                'email' => $customer?->getEmail(),
            ],
            'solicitation_parameters' => [
                'delay' => $this->delay,
                'delay_product' => $this->delay,
                'purchase_event_type' => 'BRAND_AND_PRODUCT',
            ],
            'products' => $products,
            'sales_channel' => [
                'channel' => 'online',
            ],
        ];
    }
}
