<?php

declare(strict_types=1);

namespace Dedi\SyliusAvisVerifiesPlugin\Import;

use Dedi\SyliusAvisVerifiesPlugin\Factory\ProductReviewFactoryInterface;
use Dedi\SyliusAvisVerifiesPlugin\File\ValueObject\ImportFile;
use Dedi\SyliusAvisVerifiesPlugin\Repository\AvisVerifiesAwareRepositoryInterface;
use Doctrine\ORM\EntityManagerInterface;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Sylius\Component\Core\Repository\CustomerRepositoryInterface;
use Sylius\Component\Core\Repository\OrderRepositoryInterface;
use Sylius\Component\Core\Repository\ProductReviewRepositoryInterface;
use Sylius\Component\Review\Model\ReviewerInterface;
use Sylius\Component\Review\Model\ReviewInterface;

class ReviewFileImporter implements ReviewFileImporterInterface
{
    public function __construct(
        private readonly ProductReviewRepositoryInterface $productReviewRepository,
        private readonly ProductReviewFactoryInterface $productReviewFactory,
        private readonly AvisVerifiesAwareRepositoryInterface $avisVerifiesAwareRepository,
        private readonly OrderRepositoryInterface $orderRepository,
        private readonly CustomerRepositoryInterface $customerRepository,
        private readonly EntityManagerInterface $entityManager,
        private readonly LoggerInterface $logger,
    ) {
    }

    public function import(ImportFile $importFile): void
    {
        $xml = \simplexml_load_file($importFile->getLocalFile()->getPathname());

        $i = 0;

        foreach ($xml->review as $reviewNode) {
            try {
                $review = $this->findOrCreateReview($reviewNode);
            } catch (\Exception $e) {
                $this->logger->error($e->getMessage());

                continue;
            }

            /** @var ProductInterface $reviewable */
            $reviewable = $review->getReviewSubject();

            $this->logger->info(sprintf(
                'Create/update review for customer id %d and product id %d',
                $review->getAuthor()->getId(),
                $reviewable->getId(),
            ));

            $this->entityManager->persist($review);

            ++$i;
            if ($i % 20 === 0) {
                $this->entityManager->flush();
                $i = 0;
            }
        }

        $this->entityManager->flush();
    }

    private function findOrCreateReview(\SimpleXMLElement $reviewNode): ReviewInterface
    {
        $productId = (int) $reviewNode->product_ref;
        $orderNumber = (string) $reviewNode->order_ref;
        $email = isset($reviewNode->email) ? (string) $reviewNode->email : '';

        $product = $this->avisVerifiesAwareRepository->findOneByAvisVerifiesId($productId);
        if (null === $product) {
            throw new \Exception(sprintf('Unable to find product with avisVerifiesId %d', $productId));
        }

        $customer = $this->getReviewer($orderNumber, $email);
        if (null === $customer) {
            throw new \Exception(sprintf(
                'Unable to find customer for order number %d and email %s',
                $orderNumber,
                $email,
            ));
        }

        /** @var ReviewInterface|null $review */
        $review = $this->productReviewRepository->findOneBy([
            'reviewSubject' => $product->getId(),
            'author' => $customer->getId(),
        ]);

        if (null === $review) {
            return $this->productReviewFactory->createFromAvisVerifieFeed($product, $customer, $reviewNode);
        }

        $date = new \DateTime((string) $reviewNode->review_date);
        $review->setComment((string) $reviewNode->review);
        $review->setRating((int) $reviewNode->rate);
        $review->setUpdatedAt($date);

        $attributes = $reviewNode->attributes();
        $review->setStatus(\in_array((string) $attributes['action'], ['NEW', 'UPDATE'], true)
            ? ReviewInterface::STATUS_ACCEPTED
            : ReviewInterface::STATUS_REJECTED)
        ;

        return $review;
    }

    private function getReviewer(string $orderNumber, string $email = ''): ?ReviewerInterface
    {
        /** @var OrderInterface|null $order */
        $order = $this->orderRepository->findOneByNumber($orderNumber);

        if (null !== $order) {
            /** @var ReviewerInterface $reviewer */
            $reviewer = $order->getCustomer();

            return $reviewer;
        }

        if ('' === $email) {
            return null;
        }

        /** @var ReviewerInterface|null $customer */
        $customer = $this->customerRepository->findOneBy(['email' => $email]);

        return $customer;
    }
}
