<?php

declare(strict_types=1);

namespace Dedi\SyliusAvisVerifiesPlugin\Request;

use Dedi\SyliusAvisVerifiesPlugin\Client\Factory\CredentialsFactoryInterface;
use Dedi\SyliusAvisVerifiesPlugin\Client\HttpClientInterface;
use Dedi\SyliusAvisVerifiesPlugin\Client\ValueObject\CredentialsInterface;
use Dedi\SyliusAvisVerifiesPlugin\Model\AvisVerifiesAwareInterface;
use Liip\ImagineBundle\Service\FilterService;
use Psr\Log\LoggerInterface;
use Sylius\Component\Core\Model\CustomerInterface;
use Sylius\Component\Core\Model\ImageInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Model\OrderItemInterface;
use Sylius\Component\Core\Model\ProductInterface;
use Symfony\Component\Routing\RouterInterface;
use Webmozart\Assert\Assert;

final class SendOrderRequest implements SendOrderRequestInterface
{
    private const QUERY_NAME = 'pushCommandeSHA1';

    public function __construct(
        private readonly HttpClientInterface $client,
        private readonly RouterInterface $router,
        private readonly LoggerInterface $logger,
        private readonly FilterService $imageFilterService,
        private readonly CredentialsFactoryInterface $credentialsFactory,
        private readonly string $url,
        private readonly int $delay = 0,
    ) {
    }

    public function send(OrderInterface $order): void
    {
        try {
            $credentials = $this->credentialsFactory->create();
            $message = $this->createMessage($order);
            $message['sign'] = $this->hash($credentials, $order);

            $body = [
                'idWebsite' => $credentials->getWebsiteId(),
                'message' => \json_encode($message, \JSON_THROW_ON_ERROR),
            ];

            $this->client->post($this->url, $body);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
        }
    }

    private function createMessage(OrderInterface $order): array
    {
        $products = array_map(
            function (OrderItemInterface $orderItem) use ($order): array {
                /** @var ProductInterface|AvisVerifiesAwareInterface $product */
                $product = $orderItem->getProduct();

                Assert::isInstanceOf($product, ProductInterface::class);
                Assert::isInstanceOf($product, AvisVerifiesAwareInterface::class);

                $productData = [
                    'id_product' => $product->getAvisVerifiesId(),
                    'name_product' => $product->getName(),
                    'url_product' => $this->router->generate(
                        'sylius_shop_product_show',
                        ['slug' => $product->getSlug(), '_locale' => $order->getLocaleCode()],
                        RouterInterface::ABSOLUTE_URL,
                    ),
                ];

                /** @var ImageInterface|bool $image */
                $image = $product->getImagesByType('main')->first();
                if ($image !== false) {
                    $productData['url_image_product'] = $this->imageFilterService->getUrlOfFilteredImage(
                        $image->getPath(),
                        'sylius_shop_product_thumbnail',
                    );
                }

                return $productData;
            },
            $order->getItems()->toArray(),
        );

        $customer = $order->getCustomer();

        return [
            'query' => self::QUERY_NAME,
            'order_ref' => $order->getNumber(),
            'email' => $customer?->getEmail(),
            'order_date' => $order->getCheckoutCompletedAt()?->format('Y-m-d H:i:s'),
            'firstname' => $customer?->getFirstName(),
            'lastname' => $customer?->getLastName(),
            'delay' => $this->delay,
            'sign' => null,
            'PRODUCTS' => $products,
        ];
    }

    private function hash(CredentialsInterface $credentials, OrderInterface $order): string
    {
        /** @var CustomerInterface|null $customer */
        $customer = $order->getCustomer();

        if (null === $customer) {
            throw new \Exception(sprintf('No customer for order %d', $order->getId()));
        }

        $string = sprintf(
            '%s%s%s%s%s%s%d%s',
            self::QUERY_NAME,
            $order->getNumber(),
            $customer->getEmail(),
            $customer->getLastName(),
            $customer->getFirstName(),
            $order->getCheckoutCompletedAt()?->format('Y-m-d H:i:s'),
            $this->delay,
            $credentials->getSecretKey(),
        );

        return sha1($string);
    }
}
