<?php

declare(strict_types=1);

namespace Dedi\SyliusAvisVerifiesPlugin\File;

use Dedi\SyliusAvisVerifiesPlugin\File\Factory\ImportFileFactoryInterface;
use Dedi\SyliusAvisVerifiesPlugin\File\ValueObject\ImportFile;
use Gaufrette\Adapter\Ftp;
use Gaufrette\FilesystemInterface;
use Symfony\Component\Filesystem\Filesystem;
use Webmozart\Assert\Assert;

class FileImporter implements FileImporterInterface
{
    public function __construct(
        private readonly FilesystemInterface $remoteFileSystem,
        private readonly Filesystem $localFileSystem,
        private readonly ImportFileFactoryInterface $importFileFactory,
        private readonly string $backupDirectory,
    ) {
    }

    /**
     * @throws \Exception
     */
    public function import(string $filePrefix, bool $sortDescending = true): ImportFile
    {
        $filesByMtime = $this->getRemoteFiles($filePrefix, $sortDescending);

        $file = $this->createImportFile(current($filesByMtime));

        $this->closeAdapter();

        return $file;
    }

    /**
     * @return ImportFile[]|array
     *
     * @throws \Exception
     */
    public function importAll(string $filePrefix): array
    {
        $filesByMtime = $this->getRemoteFiles($filePrefix, false);

        $files = [];

        foreach ($filesByMtime as $fileName) {
            $files[] = $this->createImportFile($fileName);
        }

        $this->closeAdapter();

        return $files;
    }

    public function archiveFile(ImportFile $file): void
    {
        $this->remoteFileSystem->rename(
            $file->getRemoteFilePath(),
            $this->backupDirectory . \DIRECTORY_SEPARATOR . basename($file->getRemoteFilePath()),
        );
        $this->localFileSystem->remove($file->getLocalFile()->getRealPath());

        $this->closeAdapter();
    }

    /**
     * @throws \Exception
     */
    private function getRemoteFiles(string $filePrefix, bool $sortDescending = true): array
    {
        $keys = $this->remoteFileSystem->listKeys($filePrefix);
        if (!array_key_exists('keys', $keys) || \count($keys['keys']) === 0) {
            throw new \Exception(sprintf('No file found with prefix %s', $filePrefix));
        }

        $filesByMtime = [];
        foreach ($keys['keys'] as $file) {
            $mtime = $this->remoteFileSystem->mtime($file);
            $filesByMtime[$mtime] = $file;
        }

        if ($sortDescending) {
            krsort($filesByMtime);
        } else {
            ksort($filesByMtime);
        }

        return $filesByMtime;
    }

    private function createImportFile(string $remoteFilePath): ImportFile
    {
        $file = $this->remoteFileSystem->get($remoteFilePath);
        $content = $file->getContent();

        $localFilePath = '/tmp/' . $file->getName();
        $this->localFileSystem->dumpFile($localFilePath, $content);

        return $this->importFileFactory->create($file->getName(), $localFilePath);
    }

    private function closeAdapter(): void
    {
        Assert::isInstanceOf($this->remoteFileSystem, \Gaufrette\Filesystem::class);

        $adapter = $this->remoteFileSystem->getAdapter();

        if (!$adapter instanceof Ftp) {
            return;
        }

        $adapter->close();
    }
}
