<?php

declare(strict_types=1);

namespace Dedi\SyliusAclPlugin\Resolver;

use Dedi\Contracts\SyliusPluginSettings\SettingsProviderInterface;
use Dedi\SyliusAclPlugin\AuthorizationChecker\AuthorizationCheckerInterface;
use Dedi\SyliusAclPlugin\PermissionMap\PermissionMapInterface;
use Dedi\SyliusAclPlugin\Provider\AdminUserProviderInterface;
use Symfony\Component\HttpFoundation\RequestMatcherInterface;
use Symfony\Component\HttpFoundation\RequestStack;

final class PermissionResolver implements PermissionResolverInterface
{
    public function __construct(
        private AuthorizationCheckerInterface $authorizationChecker,
        private PermissionMapInterface $compositePrivilege,
        private AdminUserProviderInterface $adminUserProvider,
        private RequestMatcherInterface $adminRequestMatcher,
        private RequestStack $requestStack,
        private SettingsProviderInterface $settingsProvider,
    ) {
    }

    public function hasPermission(string $permission): bool
    {
        $settings = $this->settingsProvider->provide();

        if (!$settings->isEnabled()) {
            return true;
        }

        $request = $this->requestStack->getCurrentRequest();

        if (null === $request) {
            return true;
        }

        if (!$this->adminRequestMatcher->matches($request)) {
            return true;
        }

        $supportedRouteMap = $this->compositePrivilege->getMap();

        if (!array_key_exists($permission, $supportedRouteMap)) {
            return true;
        }

        $adminUser = $this->adminUserProvider->provide();

        if (null === $adminUser) {
            return false;
        }

        if (!$adminUser->hasPermissionChecker() ||
            true === $this->authorizationChecker->isGranted($permission)
        ) {
            return true;
        }

        return false;
    }
}
