<?php

declare(strict_types=1);

namespace Dedi\SyliusAclPlugin\PermissionMap;

use Dedi\SyliusAclPlugin\Route\Checker\RouteCheckerInterface;
use Dedi\SyliusAclPlugin\Route\Label\RouteLabelResolverInterface;
use Doctrine\Inflector\Inflector;
use Doctrine\Inflector\InflectorFactory;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouterInterface;
use Webmozart\Assert\Assert;

final class ResourcePermissionMap implements PermissionMapInterface
{
    private ?Inflector $inflector = null;

    public function __construct(
        private RouterInterface $router,
        private RouteCheckerInterface $routeChecker,
        private RouteLabelResolverInterface $routeLabelResolver,
        private array $permissions,
    ) {
    }

    public function getMap(array $mapping = [], bool $isTree = false): array
    {
        foreach ($this->router->getRouteCollection() as $routeName => $route) {
            if (!$route->hasDefault('_sylius')) {
                continue;
            }

            $syliusParameters = $route->getDefault('_sylius');
            if (!is_array($syliusParameters) || !array_key_exists('permission', $syliusParameters)) {
                continue;
            }

            if (false === (bool) $syliusParameters['permission']) {
                continue;
            }

            $controller = $route->getDefault('_controller');
            if (!is_string($controller) || false !== stristr($controller, '\\')) {
                continue;
            }

            if ($this->routeChecker->check($route)) {
                $mapping = $this->addRoute($routeName, $mapping, $route, $isTree);
            }
        }

        return $mapping;
    }

    private function addRoute(
        string $routeName,
        array $currentRouteMap,
        Route $route,
        bool $isTree = false,
    ): array {
        if (isset($this->permissions[$routeName]['enabled']) && !$this->permissions[$routeName]['enabled']) {
            return $currentRouteMap;
        }

        if (
            null !== $this->getPluginParameter($route, 'enabled') &&
            false === (bool) $this->getPluginParameter($route, 'enabled')
        ) {
            return $currentRouteMap;
        }

        if (!$isTree) {
            $currentRouteMap[$routeName] = $this->getRouteLabel($routeName, $route);

            return $currentRouteMap;
        }

        $controller = $route->getDefault('_controller');
        Assert::string($controller);
        $alias = $this->getAlias($controller);
        $group = $this->getInflector()->pluralize($alias);

        $routeGroup = $this->getRouteGroup($route, $routeName);
        if (null !== $routeGroup) {
            $group = $routeGroup;
        }

        if (!isset($currentRouteMap[$group])) {
            $currentRouteMap[$group] = [];
        }

        $currentRouteMap[$group][$routeName] = $this->getRouteLabel($routeName, $route);

        return $currentRouteMap;
    }

    private function getRouteGroup(Route $route, string $routeName): ?string
    {
        if (array_key_exists($routeName, $this->permissions) &&
            array_key_exists('group', $this->permissions[$routeName]) &&
            null !== $this->permissions[$routeName]['group']
        ) {
            return $this->permissions[$routeName]['group'];
        }

        if (null !== $group = $this->getPluginParameter($route, 'group')) {
            return $group;
        }

        return null;
    }

    private function getRouteLabel(
        string $routeName,
        Route $route,
    ): ?string {
        if (array_key_exists($routeName, $this->permissions) && null !== $this->permissions[$routeName]['label']) {
            return $this->permissions[$routeName]['label'];
        }

        return $this->routeLabelResolver->resolve($routeName, $route);
    }

    private function getPluginParameter(Route $route, string $key): ?string
    {
        $defaults = $route->getDefault('_dedi_sylius_acl_plugin');
        if (\is_array($defaults) && array_key_exists($key, $defaults)) {
            return $defaults[$key];
        }

        return null;
    }

    private function getAlias(string $controller): string
    {
        if (!str_contains($controller, ':')) {
            return explode('.', $controller)[0];
        }

        return explode(':', explode('.', $controller)[2])[0];
    }

    private function getInflector(): Inflector
    {
        if (null === $this->inflector) {
            $inflectorFactory = InflectorFactory::create();
            $this->inflector = $inflectorFactory->build();
        }

        return $this->inflector;
    }
}
