<?php

declare(strict_types=1);

namespace Dedi\SyliusAclPlugin\PermissionMap;

use Dedi\SyliusAclPlugin\Exception\MissingConfigurationException;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouterInterface;

final class CustomPermissionMap implements PermissionMapInterface
{
    public function __construct(
        private RouterInterface $router,
        private array $permissions,
    ) {
    }

    public function getMap(array $mapping = [], bool $isTree = false): array
    {
        $customPermissions = $this->getCustomPermissionsWithRouteCollection($mapping, $isTree);

        foreach ($this->permissions as $permission => $config) {
            if (!$this->hasPermission($permission, $mapping, $isTree)) {
                $customPermissions[$permission] = $config;
            }
        }

        foreach ($customPermissions as $customPermission => $config) {
            if (isset($config['enabled']) && !$config['enabled']) {
                continue;
            }

            if (!array_key_exists('label', $config) || null === $config['label']) {
                throw new MissingConfigurationException($customPermission, 'label');
            }

            if (!$isTree) {
                $mapping[$customPermission] = $config['label'];

                continue;
            }

            if (!array_key_exists('group', $config) || null === $config['group']) {
                throw new MissingConfigurationException($customPermission, 'group');
            }

            if (!array_key_exists($config['group'], $mapping)) {
                $mapping[$config['group']] = [];
            }

            $mapping[$config['group']][$customPermission] = $config['label'];
        }

        ksort($mapping);

        return $mapping;
    }

    private function getCustomPermissionsWithRouteCollection(array $currentRouteMap, bool $isTree): array
    {
        $customPermissions = [];

        /** @var Route $route */
        foreach ($this->router->getRouteCollection() as $key => $route) {
            if (!$route->hasDefault('_dedi_sylius_acl_plugin') ||
                $this->hasPermission($key, $currentRouteMap, $isTree)
            ) {
                continue;
            }

            $config = $route->getDefault('_dedi_sylius_acl_plugin');

            $customPermissions[$key] = $config;
        }

        return $customPermissions;
    }

    private function hasPermission(
        string $permission,
        array $currentRouteMap,
        bool $isTree = false,
    ): bool {
        if (!$isTree) {
            return array_key_exists($permission, $currentRouteMap);
        }

        foreach ($currentRouteMap as $item) {
            if (array_key_exists($permission, $item)) {
                return true;
            }
        }

        return false;
    }
}
