<?php

declare(strict_types=1);

namespace Dedi\SyliusAclPlugin\EventListener;

use Dedi\SyliusAclPlugin\Route\Checker\RouteCheckerInterface;
use Doctrine\DBAL\Exception\ForeignKeyConstraintViolationException;
use Sylius\Component\Resource\ResourceActions;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\Flash\FlashBagInterface;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\Routing\RouterInterface;
use Webmozart\Assert\Assert;

final class ResourceDeleteSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private RouterInterface $router,
        private RequestStack $requestStack,
        private RouteCheckerInterface $adminSectionRouteChecker,
    ) {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            KernelEvents::EXCEPTION => 'onResourceDelete',
        ];
    }

    public function onResourceDelete(ExceptionEvent $event): void
    {
        $exception = $event->getThrowable();
        if (!$exception instanceof ForeignKeyConstraintViolationException) {
            return;
        }

        $request = $event->getRequest();

        if (!$event->isMainRequest() || $request->isXmlHttpRequest()) {
            return;
        }

        $eventRequest = $request;
        $requestAttributes = $eventRequest->attributes;
        $originalRoute = $requestAttributes->get('_route');
        Assert::string($originalRoute);

        $route = $this->router->getRouteCollection()->get($originalRoute);
        Assert::notNull($route);

        if (!$this->isMethodDelete($eventRequest) ||
            !$this->isPluginRoute($originalRoute) ||
            !$this->adminSectionRouteChecker->check($route)
        ) {
            return;
        }

        $resourceName = $this->getResourceNameFromRoute($originalRoute);

        if (null === $requestAttributes->get('_controller')) {
            return;
        }

        $session = $this->requestStack->getSession();

        /** @var FlashBagInterface $flashBag */
        $flashBag = $session->getBag('flashes');
        $flashBag->add('error', [
            'message' => 'sylius.resource.delete_error',
            'parameters' => ['%resource%' => $resourceName],
        ]);

        $referrer = $eventRequest->headers->get('referer');

        if (null !== $referrer) {
            $event->setResponse(new RedirectResponse($referrer));

            return;
        }

        $event->setResponse($this->createRedirectResponse($originalRoute));
    }

    private function getResourceNameFromRoute(string $route): string
    {
        $route = str_replace(['_bulk', 'acl_plugin'], '', $route);
        $routeArray = explode('_', $route);
        $routeArrayWithoutAction = array_slice($routeArray, 0, count($routeArray) - 1);
        $routeArrayWithoutPrefixes = array_slice($routeArrayWithoutAction, 2);

        return trim(implode(' ', $routeArrayWithoutPrefixes));
    }

    private function createRedirectResponse(string $route): RedirectResponse
    {
        $redirectRoute = str_replace(ResourceActions::DELETE, ResourceActions::INDEX, $route);

        return new RedirectResponse($this->router->generate($redirectRoute));
    }

    private function isMethodDelete(Request $request): bool
    {
        return Request::METHOD_DELETE === $request->getMethod();
    }

    private function isPluginRoute(string $route): bool
    {
        return str_starts_with($route, 'dedi_sylius_acl');
    }
}
