# Dedi Sylius ACL Plugin

## Installation

Install with composer

```bash
$ composer require dedi/sylius-acl-plugin
```

Add plugin to your `config/bundles.php`:

```php
return [
    ...

    Dedi\SyliusAclPlugin\DediSyliusAclPlugin::class => ['all' => true]
];
```

Import config in `config/packages/dedi_sylius_acl_plugin.yaml`:

```yaml
# config/packages/dedi_sylius_acl_plugin.yaml

imports:
    - { resource: "@DediSyliusAclPlugin/config/config.yml" }
```

Import routing in `config/routes/dedi_sylius_acl_plugin.yaml` file:

```yaml

# config/routes/dedi_sylius_acl_plugin.yaml
...

dedi_sylius_acl_plugin:
    resource: "@DediSyliusAclPlugin/config/admin_routing.yml"
```

Set up the authorization checker in the `config/packages/_sylius.yaml` configuration file:

```yaml
# config/packages/_sylius.yaml
...

sylius_resource:
    authorization_checker: dedi_sylius_acl_plugin.authorization_checker.resource
```

Override the admin user model

1. Add traits and interface to your AdminUser entity class.

```php
<?php

declare(strict_types=1);

namespace App\Entity\User;

use Dedi\SyliusAclPlugin\Model\AdminUserInterface;
use Dedi\SyliusAclPlugin\Model\PermissionCheckerTrait;
use Dedi\SyliusAclPlugin\Model\RoleAwareTrait;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Sylius\Component\Core\Model\AdminUser as BaseAdminUser;
use Symfony\Component\Validator\Constraints\Collection;

/**
 * @ORM\Entity
 * @ORM\Table(name="sylius_admin_user")
 */
class AdminUser extends BaseAdminUser implements AdminUserInterface
{
    use RoleAwareTrait {
        __construct as protected initializeAclRoles;
    }
    use PermissionCheckerTrait;

    /**
     * @ORM\Column(type="boolean", name="enabled_permission")
     */
    protected bool $permissionChecker = false;

    /**
     * @var Collection<int RoleInterface>
     * @ORM\ManyToMany(targetEntity="Dedi\SyliusAclPlugin\Entity\Role")
     * @ORM\JoinTable(
     *      name="dedi_acl_admin_users_roles",
     *      joinColumns={@ORM\JoinColumn(name="admin_user_id", referencedColumnName="id", unique=false)},
     *      inverseJoinColumns={@ORM\JoinColumn(name="acl_role_id", referencedColumnName="id", unique=false)}
     * )
     */
    protected Collection $aclRoles;

    public function __construct()
    {
        parent::__construct();

        $this->initializeAclRoles();
    }
}
```

2. Update AdminUser resource:

```yaml
# config/packages/_sylius.yaml

sylius_user:
    resources:
        admin:
            user:
                classes:
                    model: App\Entity\AdminUser
```

Add twig extension override in `config/services.yaml`:

```yaml
# config/services.yaml

services:
    twig.extension.httpkernel:
        class: Dedi\SyliusAclPlugin\Twig\HttpKernelExtension
        arguments:
            - '@twig.runtime.httpkernel'
```

Overwrite following templates in your project with templates under `/tests/Application/templates/bundles`

- @SyliusAdminBundle/AdminUser/_form.html.twig
- @SyliusUi/Grid/_default.html.twig


Run the following commands:

```
$ bin/console doctrine:migrations:diff
$ bin/console doctrine:migrations:migrate
$ bin/console assets:install
$ bin/console cache:clear
```

## Usage

### How to add a new permission?

Add permission in plugin configuration:

```yaml
# config/packages/dedi_sylius_acl_plugin.yaml
...

dedi_sylius_acl:
    permissions:
        app_admin_product_export:
            enabled: true
            group: products
            label: export
```

Add permission in route configuration:

```yaml
# config/routes/admin_routes.yaml
...
         
app_admin_product_export:
    path: /admin/products/export
    methods: [GET]
    defaults:
        _dedi_sylius_acl_plugin:
            enabled: true
            group: products
            label: export
```

For this permission you will need to add translations:

- `dedi_sylius_acl_plugin.group.products`
- `dedi_sylius_acl_plugin.action.export`

### Modify a permission

Modify an existing permission

```yaml
# config/packages/dedi_sylius_acl_plugin.yaml

dedi_sylius_acl:
    permissions:
        sylius_admin_product_index:
            group: products
            label: index
```

Modify the permission on the related route

```yaml
# config/routes/admin_routes.yaml
...

sylius_admin_product_index:
    path: /admin/products
    defaults:
        _dedi_sylius_acl_plugin:
            group: products
            label: index
```

### Disable a permission

Using plugin configuration:

```yaml
# config/packages/dedi_sylius_acl_plugin.yaml

dedi_sylius_acl:
    permissions:
        sylius_admin_product_index:
            enabled: false
```

Using the route configuration:

```yaml
# config/routes/admin_routes.yaml
...

sylius_admin_product_index:
    path: /admin/products
    defaults:
        _dedi_sylius_acl_plugin:
            enabled: false
```

### Check for permission in templates

```twig
{% if dedi_acl_has_permission("sylius_admin_product_export") %}
    <a href="{{ path('app_admin_product_export') }}">Export</a>
{% endif %}
```

